﻿using HIPS.Web.Components.Common;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.ServiceModel;

using HIPS.Client.Proxy;
using HIPS.CommonSchemas;

namespace HIPS.Web.Data.Hips
{

    /// <summary>
    /// Implements a repository for interacting with patients.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
    /// </history>
    public class PatientRepository : RepositoryBase<PatientProxy>, HIPS.Web.ModelInterface.Common.IPatientRepository
    {

        #region Properties

        /// <summary>
        /// Gets or sets an integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="31 January 2014">Initial version.</change>
        /// </history>
        public int DaysDischarged { get; private set; }

        #endregion

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        public PatientRepository() : this(0, new HIPS.Web.Components.Cache.NoCachingCacheProvider(), null)
        {
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="daysDischarged">Integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".</param>
        /// <param name="cacheProvider">Cache provider to be employed by the repository.</param>
        /// <param name="cacheKeyPrefix">Key prefix to be employed for caching.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        public PatientRepository(int daysDischarged, ICacheProvider cacheProvider, string cacheKeyPrefix = "") : base(cacheProvider, cacheKeyPrefix)
        {
            //Initialise client proxy ready for use.
            this.ClientProxy = new PatientProxy("PatientEndPoint");
            this.DaysDischarged = daysDischarged;
        }

        #endregion

        #region Methods

        /// <summary>
        /// Retrieves a list of patients who are currently admitted to (and have not been discharged from) a specified hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientListResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        public ServiceResponse<PatientSchemas.PatientListResponse> ListPatientsCurrentlyInHospital(string hospitalCodeSystem, string hospitalCode, bool? withPcehr, CommonSchemas.UserDetails operatingUser)
        {
            return CacheProvider.GetOrSetIf
            (
                this.GetMethodInvocationFullCacheKey(
                    System.Reflection.MethodInfo.GetCurrentMethod().Name,
                    new object[] { hospitalCodeSystem, hospitalCode, withPcehr, this.DaysDischarged }),
                () => this._ListPatientsCurrentlyInHospital(hospitalCodeSystem, hospitalCode, withPcehr, this.DaysDischarged, operatingUser),
                response => response.IsSuccessful
            );
        }

        #region Private

        /// <summary>
        /// Retrieves a list of patients who are currently admitted to (and have not been discharged from) a specified hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="daysDischarged">Integer value representing the number of days since the patient has been discharged to still consider a patient to be "currently in hospital".</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientListResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private ServiceResponse<PatientSchemas.PatientListResponse> _ListPatientsCurrentlyInHospital(string hospitalCodeSystem, string hospitalCode, bool? withPcehr, int daysDischarged, CommonSchemas.UserDetails operatingUser)
        {
            var result = this.ClientProxy.ListPatientsInHospital(operatingUser, hospitalCodeSystem, hospitalCode, withPcehr, daysDischarged);

            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to retrieve patients for the specified hospital.", MessageLevel.Error);
            }

            return new ServiceResponse<PatientSchemas.PatientListResponse>(result, isSuccessful, messages);
        }

        #endregion

        #endregion

    }

}
